local saveutil = {}

local lume = require"util/lume"

-- When on the main menu, just get the highest progressed character you have available.
-- This value is used when joining games.
-- Update whenever a new save slot is picked.
function saveutil.SetProgressionAsHighestForActiveSaveSlot()
	local best_overall_progression = 0

	for slot_num, character in ipairs(TheSaveSystem:GetActiveCharacterSaves()) do
		local progression = saveutil.GetProgressionForCharacterSlot(character)
		if progression > best_overall_progression then
			best_overall_progression = progression
		end
	end

	TheNet:SetProgression(best_overall_progression)
end

-- Get the highest progression of the player saves that are actually being used.
-- This value is used when hosting games.
-- Update whenever a local player joins or leaves, and whenever a new frenzy is completed.
function saveutil.SetProgressionAsHighestForActiveLocalPlayers()
	if not TheNet:IsHost() then 
		return
	end

	local best_local_progression = 0

	local local_players = TheNet:GetLocalPlayerList()

	for _, id in ipairs(local_players) do
		local character_slot = TheSaveSystem:GetCharacterForPlayerID(id)
		if character_slot then
			local character_save = TheSaveSystem:GetSaveForCharacterSlot(character_slot)
			if character_save then
				local progression = saveutil.GetProgressionForCharacterSlot(character_save)

				if progression > best_local_progression then
					best_local_progression = progression
				end
			end
		end
	end

	TheNet:SetProgression(best_local_progression)
end

function saveutil.GetProgressionForCharacterSlot(character_slot)
	local best_character_progression = 0
	local ascension_data = character_slot.persistdata.playerdata and character_slot.persistdata.playerdata.Ascension

	if not ascension_data then
		return best_character_progression
	end

	local biomes = require"defs/biomes"
	local location_unlock_order = {}

	for _, unlocks in ipairs(biomes.location_unlock_order) do
		for _, unlock in ipairs(unlocks) do
			table.insert(location_unlock_order, unlock.id)
		end
	end

	location_unlock_order = lume.invert(location_unlock_order)

	for location_id, levels in pairs(ascension_data) do

		local dungeon_score = location_unlock_order[location_id]
		local highest_frenzy = 0

		for weapon, frenzy in pairs(levels) do
			if frenzy > highest_frenzy then
				highest_frenzy = frenzy
			end
		end

		local dungeon_progression = dungeon_score + highest_frenzy

		-- printf(">> %s F%s = %s", location_id, highest_frenzy, dungeon_progression)

		if dungeon_progression > best_character_progression then
			best_character_progression = dungeon_progression
		end
	end

	-- printf(">> Slot Progression Score: %s", best_character_progression)

	return best_character_progression
end

function saveutil.CheckEnoughDiskSpace()
	local ConfirmDialog = require "screens.dialogs.confirmdialog"

	local disk_space_warning = TheSim:GetFreeDiskSpace() < 250*1024*1024
	local disk_space_critical = TheSim:HasEnoughFreeDiskSpace() == false
	if disk_space_critical then
		local dialog = ConfirmDialog(nil, nil, true, 
			STRINGS.UI.DISKSPACE.CRITICAL_TITLE,
			"",
			STRINGS.UI.DISKSPACE.CRITICAL_DESC
		)

		dialog
			:SetYesButton(STRINGS.UI.DISKSPACE.TRY_AGAIN,
				function()
					dialog:Close()
					TheFrontEnd.screenroot.inst:DoTaskInTime(0.5, function() 
						if TheSim:HasEnoughFreeDiskSpace() then
							TheSaveSystem:SaveAll()
						end

						saveutil.CheckEnoughDiskSpace() 
					end)
				end)
			:SetNoButton(STRINGS.UI.DISKSPACE.CRITICAL_QUIT, RequestShutdown)
			:HideArrow() 
			:SetMinWidth(1000)
			:SetVerticalButtons()
			:CenterText() -- Aligns left otherwise
			:CenterButtons() -- They align left otherwise
		TheFrontEnd:PushScreen(dialog)
		dialog:AnimateIn()
	elseif disk_space_warning then
		local dialog = ConfirmDialog(nil, nil, true, 
			STRINGS.UI.DISKSPACE.WARNING_TITLE,
			"",
			STRINGS.UI.DISKSPACE.WARNING_DESC
		)
		dialog
			:SetYesButton(STRINGS.UI.DISKSPACE.TRY_AGAIN,
				function()
					dialog:Close()
					TheFrontEnd.screenroot.inst:DoTaskInTime(0.5, function() saveutil.CheckEnoughDiskSpace() end)
				end)
			:SetNoButton(STRINGS.UI.DISKSPACE.CONTINUE, 
				function() 
					dialog:Close() 
				end, true)
			:HideArrow() 
			:SetMinWidth(1000)
			:SetVerticalButtons()
			:CenterText() -- Aligns left otherwise
			:CenterButtons() -- They align left otherwise
		TheFrontEnd:PushScreen(dialog)
		dialog:AnimateIn()
	end

end

return saveutil
